package presentation; 
/**
 * @author Alberto Iachini
 */

import java.awt.BorderLayout;
import javax.swing.JPanel;
import javax.swing.JFrame;
import javax.swing.JMenuBar;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JTabbedPane;
import java.awt.Dimension;
import control.Orchestrator;
import control.Community;
import java.io.File;
import abstraction.CartesianProduct;
import abstraction.Service;
import javax.swing.JLabel;

/**
 * 
 * This class is the main windows
 *
 */
public class OrchestratorFrame extends JFrame {

	private static final long serialVersionUID = -5592078694670051572L;

	private static final String ABOUT = "Opus\n\n Course in Computer Engineering\n             \"La Sapienza di Roma\"\n       Accedemic year 2008\\2009\n"+
	"\nAuthors:\nBalestra Concetta\nCoccia Chiara\nColarullo Diego\nIachini Alberto\nQusa Hani\n"+    
	"\nService Integration\nProf. Giuseppe De Giacomo\n\n            15th May 2009\n ";  //  @jve:decl-index=0:
	
	private static final String HELP = "For more details see user manual or documentation.";
	
	private JPanel mainContentPane = null;
	private JTabbedPane mainTabbedPane = null;
	
	private AvailablePanel availablePanel = null;
	private TargetPanel targetPanel = null;
	private OrchestratorTablePanel orchestratorTablePanel = null;
	private CartesianProductPanel cartesianProductPanel = null;
	private ExecuteOrchestratorPanel executeOrchestratorPanel = null;
	
	private JMenuBar menubar = null;
	
	private JMenu fileMenu = null;
	private JMenuItem exitItem = null;
	
	private JMenu OrchestratorMenu = null;
	private JMenuItem startItem = null;
	private JMenuItem stopItem = null;
	
	private JMenu helpMenu = null;
	private JMenuItem helpItem = null;
	private JMenuItem aboutItem = null;
	
	private Orchestrator orchestrator = null;
	
	private File lastFolder;

	private JLabel warningLabel = null;

	private OrchestratorGraphPanel orchestratorGraphPanel = null;
	
	/**
	 * This is the default constructor
	 */
	public OrchestratorFrame() {
		super();
		initialize();
	}

	/**
	 * This method initializes this
	 */
	private void initialize() {
		this.setSize(800, 500);
		this.setJMenuBar(getMenubar());
		this.setContentPane(getJContentPane());
		this.setTitle("Opus");
		this.setMaximumSize(new Dimension(300,300));
		setVisible(true);
		setDefaultCloseOperation(EXIT_ON_CLOSE);
	}

	/**
	 * This method initializes jContentPane
	 * 
	 * @return javax.swing.JPanel
	 */
	private JPanel getJContentPane() {
		if (mainContentPane == null) {
			BorderLayout borderLayout = new BorderLayout();
			borderLayout.setHgap(0);
			warningLabel = new JLabel();
			warningLabel.setText("    * if there is a node with a loop, could be action's names overlapped");
			mainContentPane = new JPanel();
			mainContentPane.setLayout(borderLayout);
			mainContentPane.add(getMainTabbedPane(), BorderLayout.CENTER);
			mainContentPane.add(warningLabel, BorderLayout.SOUTH);
		}
		return mainContentPane;
	}

	/**
	 * This method initializes menubar	
	 * 	
	 * @return javax.swing.JMenuBar	
	 */
	private JMenuBar getMenubar() {
		if (menubar == null) {
			menubar = new JMenuBar();
			menubar.add(getFileMenu());
			menubar.add(getOrchestratorMenu());
			menubar.add(getHelpMenu());
		}
		return menubar;
	}

	/**
	 * This method initializes fileMenu	
	 * 	
	 * @return javax.swing.JMenu	
	 */
	private JMenu getFileMenu() {
		if (fileMenu == null) {
			fileMenu = new JMenu();
			fileMenu.setText("File");
			fileMenu.add(getExitItem());
		}
		return fileMenu;
	}

	/**
	 * This method initializes OrchestratorMenu	
	 * 	
	 * @return javax.swing.JMenu	
	 */
	private JMenu getOrchestratorMenu() {
		if (OrchestratorMenu == null) {
			OrchestratorMenu = new JMenu();
			OrchestratorMenu.setText("Orchestrator");
			OrchestratorMenu.add(getStartItem());
			OrchestratorMenu.add(getStopItem());
		}
		return OrchestratorMenu;
	}

	/**
	 * This method initializes helpMenu	
	 * 	
	 * @return javax.swing.JMenu	
	 */
	private JMenu getHelpMenu() {
		if (helpMenu == null) {
			helpMenu = new JMenu();
			helpMenu.setText("Help");
			helpMenu.add(getHelpItem());
			helpMenu.add(getAboutItem());
		}
		return helpMenu;
	}


	/**
	 * This method initializes exitItem1	
	 * 	
	 * @return javax.swing.JMenuItem	
	 */
	private JMenuItem getExitItem() {
		if (exitItem == null) {
			exitItem = new JMenuItem();
			exitItem.setText("Exit");
			exitItem.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					System.exit(0);
				}
			});
		}
		return exitItem;
	}

	/**
	 * This method initializes startItem	
	 * 	
	 * @return javax.swing.JMenuItem	
	 */
	private JMenuItem getStartItem() {
		if (startItem == null) {
			startItem = new JMenuItem();
			startItem.setText("Start");
			startItem.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					/*
					 * 1 - check if existe a target and at least one available service
					 * 2 - start  the orchestrator generator
					 * 3 - enable and initialize the panels Orchesstator and ExecuteOrchestrator
					 */
					if(!targetPanel.isTargetLoaded())
					{
						JOptionPane.showMessageDialog(null, "No Targer Service Loaded", "Error", JOptionPane.ERROR_MESSAGE);
					}
					else if(!availablePanel.isAvailableLoaded())
					{
						JOptionPane.showMessageDialog(null, "No Available Services Loaded", "Error", JOptionPane.ERROR_MESSAGE);
					}
					else
					{
						long time = System.currentTimeMillis();
						Community com = new Community(availablePanel.getServices());
						orchestrator = new Orchestrator(com, targetPanel.getTargetService());

						// if flag is false then the orchestrator does not exists
						boolean flag = orchestrator.generateOrchestrator();
						time = System.currentTimeMillis()-time;
						
						if(flag == false)
						{
							// composition and orchestrator is empty
							JOptionPane.showMessageDialog(null, 
									"Impossible to create orchestrator because composition does not exists", 
									"Warning", JOptionPane.WARNING_MESSAGE);
							
							orchestrator = null;
						}
						else 
						{
							
							mainTabbedPane.setEnabledAt(3, true);
							mainTabbedPane.setEnabledAt(4, true);
							mainTabbedPane.setEnabledAt(5, true);
							
							// if chechComposition is false then composition is empty
							if(orchestrator.checkComposition() == false)
							{
								// composition is empty 
								JOptionPane.showMessageDialog(null, 
										"Composition does not exist, orchestrator may be not run correctly", 
										"Warning", JOptionPane.WARNING_MESSAGE);
								
								mainTabbedPane.setEnabledAt(5, false);
							}
							startItem.setEnabled(false);
							stopItem.setEnabled(true);
							availablePanel.setEnableButtons(false);
							targetPanel.setEnableButtons(false);

							executeOrchestratorPanel.initExecuteOrchestratorPanel(targetPanel.getTargetService(), com, orchestrator);
							orchestratorTablePanel.init(orchestrator);
							orchestratorGraphPanel.init(orchestrator);
							
							
						}
						
						
						long ms = time%1000;
						time = time/1000;
						long sec = time%60;
						time = time/60;
						long min = time%60;
						time = time/60;
						
						String mess = "Elapsed Time: "+ time+ " hours, "
							+ min + " minutes, " 
							+ sec + " seconds, "
							+ ms + " milliseconds";
						JOptionPane.showMessageDialog(null, mess, "Time", JOptionPane.INFORMATION_MESSAGE);
					}
				}
			});
		}
		return startItem;
	}

	/**
	 * This method initializes helpItem	
	 * 	
	 * @return javax.swing.JMenuItem	
	 */
	private JMenuItem getHelpItem() {
		if (helpItem == null) {
			helpItem = new JMenuItem();
			helpItem.setText("Help");
			helpItem.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					showHelp();
				}
			});
		}
		return helpItem;
	}

	/**
	 * This method initializes aboutItem	
	 * 	
	 * @return javax.swing.JMenuItem	
	 */
	private JMenuItem getAboutItem() {
		if (aboutItem == null) {
			aboutItem = new JMenuItem();
			aboutItem.setText("About");
			aboutItem.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					showAbout();
				}
			});
		}
		return aboutItem;
	}
	
	/**
	 * This method shows the About Dialog
	 */
	private void showAbout()
	{
		JOptionPane.showMessageDialog(this, ABOUT, "About", JOptionPane.INFORMATION_MESSAGE);
	}
	
	/**
	 * This method shows the Help Dialog
	 */
	private void showHelp()
	{
		JOptionPane.showMessageDialog(this, HELP, "Help", JOptionPane.INFORMATION_MESSAGE);
	}

	/**
	 * This method initializes mainTabbedPane	
	 * 	
	 * @return javax.swing.JTabbedPane	
	 */
	private JTabbedPane getMainTabbedPane() {
		if (mainTabbedPane == null) {
			mainTabbedPane = new JTabbedPane();
			mainTabbedPane.addTab("Available Services", null, getAvailablePanel(), null);
			mainTabbedPane.addTab("Target Service", null, getTargetPanel(), null);
			mainTabbedPane.addTab("Cartesian Product", null, getCartesianProductPanel(), null);
			
			mainTabbedPane.addTab("Orchestrator Table", null, getOrchestratorPanel(), null);
			mainTabbedPane.addTab("Orchestrator Generator", null, getOrchestratorGraphPanel(), null);
			mainTabbedPane.addTab("Execute Orchestrator", null, getExecuteOrchestratorPanel(), null);
			
			mainTabbedPane.setEnabledAt(3, false);
			mainTabbedPane.setEnabledAt(4, false);
			mainTabbedPane.setEnabledAt(5, false);
			
		}
		return mainTabbedPane;
	}

	/**
	 * This method initializes availablePanel	
	 * 	
	 * @return presentation.AvailablePanel	
	 */
	private AvailablePanel getAvailablePanel() {
		if (availablePanel == null) {
			availablePanel = new AvailablePanel(this);
		}
		return availablePanel;
	}

	/**
	 * This method initializes targetPanel	
	 * 	
	 * @return presentation.TargetPanel	
	 */
	private TargetPanel getTargetPanel() {
		if (targetPanel == null) {
			targetPanel = new TargetPanel(this);
		}
		return targetPanel;
	}

	/**
	 * This method initializes orchestratorPanel	
	 * 	
	 * @return presentation.OrchestratorPanel	
	 */
	private OrchestratorTablePanel getOrchestratorPanel() {
		if (orchestratorTablePanel == null) {
			orchestratorTablePanel = new OrchestratorTablePanel(this);
		}
		return orchestratorTablePanel;
	}


	/**
	 * This method initializes cartesianProductPanel	
	 * 	
	 * @return presentation.CartesianProductPanel	
	 */
	private CartesianProductPanel getCartesianProductPanel() {
		if (cartesianProductPanel == null) {
			cartesianProductPanel = new CartesianProductPanel(this);
		}
		return cartesianProductPanel;
	}
	
	/**
	 * This method create and return the Cartesian Product
	 * 
	 * @return Service that contains the cartesian product
	 */
	public Service getProduct()
	{
		return CartesianProduct.executeProduct(availablePanel.getServices());
	}

	/**
	 * This method initializes stopItem	
	 * 	
	 * @return javax.swing.JMenuItem	
	 */
	private JMenuItem getStopItem() {
		if (stopItem == null) {
			stopItem = new JMenuItem();
			stopItem.setEnabled(false);
			stopItem.setText("Stop");
			stopItem.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					stopItem.setEnabled(false);
					startItem.setEnabled(true);
					
					availablePanel.setEnableButtons(true);
					targetPanel.setEnableButtons(true);
					//serviceBuilderPanel.setEnableButtons(true);
					
					mainTabbedPane.setEnabledAt(3, false);
					mainTabbedPane.setEnabledAt(4, false);
					mainTabbedPane.setEnabledAt(5, false);
					int selected = mainTabbedPane.getSelectedIndex();
					if(selected == 3 || selected == 4 || selected == 5)
						mainTabbedPane.setSelectedIndex(1);
				}
			});
		}
		return stopItem;
	}

	/**
	 * This method initializes simulationPanel	
	 * 	
	 * @return presentation.SimulationPanel	
	 */
	private ExecuteOrchestratorPanel getExecuteOrchestratorPanel() {
		if (executeOrchestratorPanel == null) {
			executeOrchestratorPanel = new ExecuteOrchestratorPanel(this);
		}
		return executeOrchestratorPanel;
	}
	
	/**
	 * This method check if at least one available service are loaded
	 * 
	 * @return true if services are loaded
	 */
	public boolean isAvailableLoaded()
	{
		return availablePanel.isAvailableLoaded();
	}

	/**
	 * This method return the last directory used by user
	 * 
	 * @return the lastFolder
	 */
	public File getLastFolder() {
		return lastFolder;
	}

	/**
	 * This method set the last directory used by user
	 * 
	 * @param folder the lastFolder to set
	 */
	public void setLastFolder(File folder) {
		if(folder != null)
			this.lastFolder = folder;
	}

	/**
	 * This method initializes orchestratorGraphPanel	
	 * 	
	 * @return presentation.OrchestratorGraphPanel	
	 */
	private OrchestratorGraphPanel getOrchestratorGraphPanel() {
		if (orchestratorGraphPanel == null) {
			orchestratorGraphPanel = new OrchestratorGraphPanel(this);
		}
		return orchestratorGraphPanel;
	}
}  //  @jve:decl-index=0:visual-constraint="10,10"
